package toolbus.tifs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Calendar;
import java.util.Iterator;
import java.util.StringTokenizer;

import aterm.ATerm;
import aterm.ATermFactory;

public class TifToJava{
	private final Tif tif;
	private final String outputBaseDir;
	
	private String capitalize(String s){
		return s.substring(0, 1).toUpperCase() + s.substring(1);
	}
	
	public TifToJava(Tif tif, String outputBaseDir){
		this.tif = tif;
		this.outputBaseDir = outputBaseDir;
	}
	
	private void generate(){
		Iterator<Process> iter = tif.fetchProcessIterator();
		while(iter.hasNext()){
			Process process = iter.next();
			PrintStream out = openOutputFile(process);
			emitHeader(out);
			generateProcessInterface(process, out);
			out.close();
		}
	}
	
	private void emitHeader(PrintStream out){
		String timeStamp = Calendar.getInstance().getTime().toString();
		out.println("// Generated by TifToJava on " + timeStamp);
		out.println();
		out.println("package " + tif.getPackageName() + ";");
		out.println();
	}
	
	private void generateProcessInterface(Process process, PrintStream out){
		out.println("public interface " + getInterfaceName(process));
		out.println('{');
		Iterator<Communication> iter = process.fetchCommunicationIterator();
		while(iter.hasNext()){
			Communication comm = iter.next();
			out.print("  ");
			generateCommunication(comm, out);
		}
		out.println('}');
	}
	
	private void generateCommunication(Communication comm, PrintStream out){
		out.print("public ");
		out.print(comm.getResultType());
		out.print(' ');
		out.print(communicationToMethodName(comm.getName()));
		out.print('(');
		generateArguments(comm, out);
		out.println(");");
	}
	
	private void generateArguments(Communication comm, PrintStream out){
		Iterator<String> iter = comm.fetchArgumentIterator();
		int argCount = 0;
		while(iter.hasNext()){
			String type = iter.next();
			out.print(type);
			out.print(' ');
			out.print("arg");
			out.print(argCount++);
			if(iter.hasNext()){
				out.print(", ");
			}
		}
	}
	
	private String getInterfaceName(Process process){
		return capitalize(tif.getToolName()) + process.getName();
	}
	
	private PrintStream openOutputFile(Process process){
		File outputDirectory = createOutputDirectory(tif.getPackageName());
		String fileName = getInterfaceName(process) + ".java";
		try{
			File outputFile = new File(outputDirectory, fileName);
			return new PrintStream(new FileOutputStream(outputFile));
		}catch(FileNotFoundException e){
			throw new RuntimeException(e);
		}
	}
	
	private File createOutputDirectory(String packageName){
		String outputDirName = packageName.replace('.', File.separatorChar);
		if(!outputBaseDir.equals("")){
			outputDirName = outputBaseDir + File.separatorChar + outputDirName;
		}
		
		File outputDir = new File(outputDirName);
		outputDir.mkdirs();
		return outputDir;
	}
	
	
	// Convert dash-separated communication-name to javaMethodName
	private String communicationToMethodName(String communicationName){
		StringBuilder buf = new StringBuilder();
		StringTokenizer tokenizer = new StringTokenizer(communicationName, "-");
		if(tokenizer.hasMoreTokens()){
			buf.append(tokenizer.nextToken());
		}
		while(tokenizer.hasMoreTokens()){
			buf.append(capitalize(tokenizer.nextToken()));
		}
		return buf.toString();
	}
	
	private static boolean isCommandLineSwitch(String arg, String shortName, String longName){
		return shortName.equals(arg) || longName.startsWith(arg);
	}
	
	public static void main(String[] args){
		String fileName = "";
		String outputBaseDir = "";
		
		for(int i = 0; i < args.length; i++){
			if(isCommandLineSwitch(args[i], "-t", "--tifs")){
				fileName = args[++i];
			}
			if(isCommandLineSwitch(args[i], "-o", "--output-dir")){
				outputBaseDir = args[++i];
			}
		}
		
		ATermFactory factory = new aterm.pure.PureFactory();
		ATerm tifsTerm = null;
		
		try{
			tifsTerm = factory.readFromFile(fileName);
		}catch(IOException e){
			System.err.println("Error reading " + fileName + ": " + e);
		}
		Tif tif = new Tif(tifsTerm);
		// System.out.println(tif.toString());
		TifToJava generator = new TifToJava(tif, outputBaseDir);
		generator.generate();
	}
	
}
